<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\ResponseData;
use App\Http\Controllers\Controller;
use App\Http\Resources\Blog\BlogResource;
use App\Models\Blog;
use App\Models\BlogCategory;
use App\Models\BlogSpecification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Illuminate\Validation\Rule;

class BlogController extends Controller
{

    public function index()
    {
        $blogs = Blog::query();

        if ($keyword = request('search')) {
            $blogs->where('title', 'LIKE', "%{$keyword}%")
                ->orWhere('description', 'LIKE', "%{$keyword}%")
                ->orWhere('keywords', 'LIKE', "%{$keyword}%");
        }

        $blogs = $blogs->orderByDesc('special')->orderByDesc('id')->paginate(20);
        return BlogResource::collection($blogs);
    }

    public function create()
    {
        //
    }

    public function edit($id)
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     */
    public function store(Request $request)
    {
        $validData = $request->validate([
            'title' => ['required', 'unique:blogs,title'],
            'description' => ['required', 'string'],
            'index_image' => ['required'],
            'index_image_alt' => ['required', 'string'],
            'keywords' => ['required', 'string', 'unique:blogs,keywords'],
            'author' => ['required', 'string'],
            'index_description' => ['required', 'string'],
            'url' => ['required', 'string', 'unique:blogs,url'],
            'seo_title' => ['string', 'required'],
            'meta_description' => ['string', 'required'],
            'og_title' => ['string', 'required'],
            'og_description' => ['string', 'required'],
            'og_type' => ['string', 'required'],
            'special' => ['string']
        ]);

        $blogCategory = BlogCategory::whereId($request->blog_category_id)->first();
        $blog = $blogCategory->blogs()->create($validData);

        $image = $request->file('index_image');
        $filename = time() . '_' . $image->getClientOriginalName();
        $image->move(public_path('blogs'), $filename);
        $photo_path = '/blogs/' . $filename;

        $og_image = $request->file('og_image');
        $og_filename = time() . '_' . $og_image->getClientOriginalName();
        $og_image->move(public_path('blogs'), $og_filename);
        $og_photo_path = '/blogs/' . $og_filename;

        $blog->index_image = $photo_path;
        $blog->og_image = $og_photo_path;
        $blog->update();

        if ($blog->special == 1) {
            Blog::where('special', 1)->where('id', '!=', $blog->id)->update(['special' => 0]);
        }

        return ResponseData::success("نوشته با موفقیت ثبت شد", new BlogResource($blog));
    }

    /**
     * Display the specified resource.
     *
     * @param Blog $blog
     */
    public function show($id)
    {
        $blog = Blog::where('id', $id)->first();
        return ResponseData::success('', new BlogResource($blog));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param Blog $blog
     */
    public function update(Request $request)
    {
        $blog = Blog::where('id', $request->id)->first();
        $validData = $request->validate([
            'title' => ['required', Rule::unique('blogs')->ignore($blog->id)],
            'description' => ['required', 'string'],
            'keywords' => ['required', 'string', Rule::unique('blogs')->ignore($blog->id)],
            'author' => ['required', 'string'],
            'index_description' => ['required', 'string'],
            'url' => ['required', 'string', Rule::unique('blogs')->ignore($blog->id)],
            'seo_title' => ['string', 'required'],
            'meta_description' => ['string', 'required'],
            'og_title' => ['string', 'required'],
            'og_description' => ['string', 'required'],
            'og_type' => ['string', 'required'],
            'index_image_alt' => ['required', 'string'],
            'special' => ['string']
        ]);

        $blog->update($validData);

        if ($request->blog_category_id) {
            $blog->blog_category_id = $request->blog_category_id;
            $blog->save();
        }

        if ($request->hasFile('index_image')) {
            $image = $request->file('index_image');
            $filename = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('blogs'), $filename);
            $photo_path = '/blogs/' . $filename;
            $blog->index_image = $photo_path;
            $blog->update();
        }

        if ($request->hasFile('og_image')) {
            $og_image = $request->file('og_image');
            $og_filename = time() . '_' . $og_image->getClientOriginalName();
            $og_image->move(public_path('blogs'), $og_filename);
            $og_photo_path = '/blogs/' . $og_filename;
            $blog->og_image = $og_photo_path;
            $blog->update();
        }

        if ($blog->special == 1) {
            Blog::where('special', 1)->where('id', '!=', $blog->id)->update(['special' => 0]);
        }
        return ResponseData::success("نوشته  با موفقیت ویرایش شد", new BlogResource($blog));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Blog $blog
     */
    public function destroy(Request $request)
    {
        $blog = Blog::where('id', $request->id)->first();
        $blog->delete();
        return ResponseData::success("نوشته با موفقیت حذف شد");
    }

    public function blogEditorImages(Request $req)
    {
        $data = array();
        if ($req->hasfile('images')) {

            foreach ($req->file('images') as $image) {
                $filename = time() . '_' . $image->getClientOriginalName();
                $image->move(public_path('editor/blogs'), $filename);
                $photo_path = URL::to('/') . '/editor/blogs/' . $filename;

                array_push($data, $photo_path);
            }
        }
        return ResponseData::success("Done", $data);
    }


    public function manageLocaleSpecification(Request $req)
    {
        $specification = BlogSpecification::where([["blog", $req->blog], ['locale', $req['locale']]])->first();
        if (!$specification) {
            $specification = BlogSpecification::create([
                'locale' => $req['locale'],
                'title' => $req->title,
                'description' => $req->description,
                'keywords' => $req->keywords,
                'author' => $req->author,
                'index_description' => $req->index_description,
                'blog' => $req->blog
            ]);


            if ($specification) {
                return ResponseData::success('تغییرات ثبت شد');
            }
        } else {
            $specification['title'] = $req->title;
            $specification['description'] = $req->description;
            $specification['keywords'] = $req->keywords;
            $specification['author'] = $req->author;
            $specification['index_description'] = $req->index_description;

            if ($specification->save()) {
                return ResponseData::success('تغییرات ثبت شد');
            } else {
                return ResponseData::error('خطا در ثبت تغییرات');
            }
        }
    }

    public function getLocaleSpecification(Request $req)
    {
        $specification = BlogSpecification::where([["blog", $req->blog], ['locale', $req['locale']]])->first();
        if (!$specification) {
            $specification['title'] = '';
            $specification['description'] = '';
            $specification['keywords'] = '';
            $specification['author'] = '';
            $specification['index_description'] = '';
        }
        return ResponseData::success('Locales', $specification);
    }
}
