<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\ResponseData;
use App\Http\Controllers\Controller;
use App\Http\Resources\Blog\BlogCategoryResource;
use App\Models\BlogCategory;
use App\Models\BlogCategorySpecification;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class BlogCategoryController extends Controller
{
    public function index()
    {
        $blogCategory = BlogCategory::query();

        $blogCategory = $blogCategory->orderByDesc('id')->paginate(20);
        return BlogCategoryResource::collection($blogCategory);
    }

    public function create()
    {
        //
    }

    public function edit($id)
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validData = $request->validate([
            'name' => ['required', 'unique:blog_categories,name'],
            'url_name' => ['required', 'string', 'unique:blog_categories,url_name'],
            'keywords' => ['required', 'string'],
            'author' => ['required', 'string'],
            'index_description' => ['required', 'string'],
            'seo_title' => ['string', 'required'],
            'meta_description' => ['string', 'required'],
            'og_title' => ['string', 'required'],
            'og_description' => ['string', 'required'],
            'og_type' => ['string', 'required'],
            'description' => ['string', 'required'],
        ]);


        if ($request->parent > 0) {
            $parent = $request->validate([
                'parent' => ['required', 'exists:blog_categories,id']
            ]);
            $validData['parent'] = $parent['parent'];
        } else
            $validData['parent'] = 0;


        $blogCategory = BlogCategory::create($validData);

        if ($request->hasFile('og_image')) {
            $og_image = $request->file('og_image');
            $og_filename = time() . '_' . $og_image->getClientOriginalName();
            $og_image->move(public_path('blogs'), $og_filename);
            $og_photo_path = '/blogs/' . $og_filename;
            $blogCategory->og_image = $og_photo_path;
            $blogCategory->update();
        }
        return ResponseData::success("دسته بندی با موفقیت ایجاد شد", new BlogCategoryResource($blogCategory));
    }

    /**
     * Display the specified resource.
     *
     * @param BlogCategory $blogCategory
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $blogCategory = BlogCategory::where('id', $id)->first();
        return ResponseData::success('', new BlogCategoryResource($blogCategory));
    }


    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param BlogCategory $blogCategory
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $blogCategory = BlogCategory::where('id', $request->id)->first();
        $validData = $request->validate([
            'name' => ['required', 'max:256',  Rule::unique('blog_categories')->ignore($request->id)],
            'url_name' => ['required', 'max:256',  Rule::unique('blog_categories')->ignore($request->id)],
            'keywords' => ['required', 'string'],
            'author' => ['required', 'string'],
            'index_description' => ['required', 'string'],
            'seo_title' => ['string', 'required'],
            'meta_description' => ['string', 'required'],
            'og_title' => ['string', 'required'],
            'og_description' => ['string', 'required'],
            'og_type' => ['string', 'required'],
            'description' => ['string', 'required'],
        ]);

        if ($request->parent) {

            if ($request->parent > 0) {
                $parent = $request->validate([
                    'parent' => ['required']
                ]);
                $validData['parent'] = $parent['parent'];
            } else
                $validData['parent'] = 0;
        }

        $blogCategory->update($validData);

        if ($request->hasFile('og_image')) {
            $og_image = $request->file('og_image');
            $og_filename = time() . '_' . $og_image->getClientOriginalName();
            $og_image->move(public_path('blogs'), $og_filename);
            $og_photo_path = '/blogs/' . $og_filename;
            $blogCategory->og_image = $og_photo_path;
            $blogCategory->update();
        }

        return ResponseData::success("دسته بندی با موفقیت ویرایش شد", new BlogCategoryResource($blogCategory));
    }

    public function destroy(Request $req)
    {
        $blogCategory = BlogCategory::where('id', $req->id)->first();
        $this->cascadingDeletionOfBlogCategory($blogCategory);

        $blogCategories = BlogCategory::whereParent(-1)->get();
        foreach ($blogCategories as $category)
            $category->delete();

        $blogCategory->delete();
        return ResponseData::success("دسته بندی حذف شد");
    }

    public function cascadingDeletionOfBlogCategory(BlogCategory $blogCategory)
    {
        $children = BlogCategory::whereParent($blogCategory->id)->get();
        foreach ($children as $child) {
            $child->parent = -1;
            $child->save();
            if ($child->child->count())
                $this->cascadingDeletionOfBlogCategory($child);
        }

        return;
    }

    public function allBlogCategories()
    {
        $allCategory = BlogCategory::all();
        return BlogCategoryResource::collection($allCategory);
    }

    public function manageLocaleSpecification(Request $req)
    {
        $specification = BlogCategorySpecification::where([["blogCategory", $req->blogCategory], ['locale', $req['locale']]])->first();
        if (!$specification) {
            $specification = BlogCategorySpecification::create([
                'locale' => $req['locale'],
                'name' => $req->name,
                'blogCategory' => $req->blogCategory
            ]);
            if ($specification) {
                return ResponseData::success("تغییرات ثبت شد");
            }
        } else {
            $specification['name'] = $req->name;

            if ($specification->save()) {
                return ResponseData::success("تغییرات ثبت شد");
            } else {
                return ResponseData::error('خطا در ثبت تغییرات');
            }
        }
    }

    public function getLocaleSpecification(Request $req)
    {
        $specification = BlogCategorySpecification::where([["blogCategory", $req->blogCategory], ['locale', $req['locale']]])->first();
        if (!$specification) {
            $specification['name'] = '';
        }
        return ResponseData::success('Locales', $specification);
    }
}
